<?php
// Activar reporte de errores y logging
error_reporting(E_ALL);
ini_set('display_errors', 1);
file_put_contents('debug.log', "=== Nuevo Request ===\n", FILE_APPEND);
file_put_contents('debug.log', "Hora: ".date('Y-m-d H:i:s')."\n", FILE_APPEND);

include 'config.php';
header('Content-Type: application/json');

// Verificar/crear directorio signed/
if (!file_exists(SIGNED_DIR)) {
    if (!mkdir(SIGNED_DIR, 0777, true)) {
        die(json_encode(['success' => false, 'message' => "No se pudo crear el directorio signed/"]));
    }
}

// Log del input recibido
$inputRaw = file_get_contents('php://input');
file_put_contents('debug.log', "Payload recibido:\n".$inputRaw."\n", FILE_APPEND);

$input = json_decode($inputRaw, true);
if (!$input || !isset($input['docId']) || !isset($input['signature'])) {
    die(json_encode(['success' => false, 'message' => 'Datos incompletos o formato incorrecto']));
}

try {
    // Validar documento
    $stmt = $db->prepare("SELECT file_path FROM documents WHERE id = ?");
    $stmt->execute([$input['docId']]);
    $document = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$document) {
        throw new Exception("Documento no encontrado en la base de datos");
    }

    file_put_contents('debug.log', "Ruta del PDF original: ".$document['file_path']."\n", FILE_APPEND);

    // Verificar existencia del PDF original
    if (!file_exists($document['file_path'])) {
        throw new Exception("El archivo PDF original no existe en: ".$document['file_path']);
    }

    // Procesar firma (convertir base64 a imagen temporal)
    $signatureFile = tempnam(sys_get_temp_dir(), 'sig');
    file_put_contents('debug.log', "Ruta temporal de firma: ".$signatureFile."\n", FILE_APPEND);

    $signatureParts = explode(',', $input['signature']);
    if (count($signatureParts) !== 2) {
        throw new Exception("Formato de firma inválido: debe incluir 'data:image/png;base64,'");
    }

    $signatureBin = base64_decode($signatureParts[1]);
    if ($signatureBin === false) {
        throw new Exception("Decodificación base64 fallida");
    }

    if (file_put_contents($signatureFile, $signatureBin) === false) {
        throw new Exception("No se pudo guardar la imagen temporal");
    }

    // Crear PDF firmado
    require_once('vendor/autoload.php');
    
    try {
        $pdf = new \setasign\Fpdi\Fpdi();
        $pdf->SetFont('Arial', '', 10);
        
        // Importar página del PDF original
        $pageCount = $pdf->setSourceFile($document['file_path']);
        if ($pageCount < 1) {
            throw new Exception("El PDF no contiene páginas válidas");
        }
        
        $templateId = $pdf->importPage(1);
        $size = $pdf->getTemplateSize($templateId);
        
        // Crear nueva página
        $pdf->AddPage($size['orientation'], [$size['width'], $size['height']]);
        $pdf->useTemplate($templateId);
        
        // Insertar firma (posición ajustable)
        $pdf->Image($signatureFile, 120, $size['height']-50, 80, 40, 'PNG');
        
    } catch (Exception $e) {
        throw new Exception("Error al procesar PDF: ".$e->getMessage());
    } finally {
        // Eliminar archivo temporal
        if (file_exists($signatureFile)) {
            unlink($signatureFile);
        }
    }

    // Guardar PDF firmado
    $signedPath = SIGNED_DIR . 'signed_' . time() . '_' . basename($document['file_path']);
    file_put_contents('debug.log', "Intentando guardar PDF firmado en: ".$signedPath."\n", FILE_APPEND);
    
    if (!is_writable(dirname($signedPath))) {
        throw new Exception("No se puede escribir en el directorio: ".dirname($signedPath));
    }

    $pdf->Output('F', $signedPath);
    
    if (!file_exists($signedPath)) {
        throw new Exception("El PDF firmado no se generó correctamente");
    }

    // Actualizar BD
    $update = $db->prepare("UPDATE documents SET signed_path = ?, status = 'signed', signed_at = NOW() WHERE id = ?");
    $updateSuccess = $update->execute([$signedPath, $input['docId']]);
    
    if (!$updateSuccess) {
        throw new Exception("Error al actualizar la base de datos");
    }

    file_put_contents('debug.log', "Proceso completado con éxito\n", FILE_APPEND);
    echo json_encode(['success' => true, 'path' => $signedPath]);

} catch (Exception $e) {
    $errorMsg = "Error: ".$e->getMessage();
    file_put_contents('debug.log', $errorMsg."\n", FILE_APPEND);
    echo json_encode(['success' => false, 'message' => $errorMsg]);
}
?>